import oracle.AWXML.AggregationDefinition;
import oracle.AWXML.AggregationHierarchySpecification;
import oracle.AWXML.Attribute;
import oracle.AWXML.AttributeProjection;
import oracle.AWXML.AttributeSourceExpression;
import oracle.AWXML.AW;
import oracle.AWXML.AWConnection;
import oracle.AWXML.Cube;
import oracle.AWXML.CubeDimensionSourceExpression;
import oracle.AWXML.CubeDimRef;
import oracle.AWXML.CubeMapGroup;
import oracle.AWXML.Dimension;
import oracle.AWXML.DimensionKeySourceExpression;
import oracle.AWXML.DimensionMapGroup;
import oracle.AWXML.DimensionMemberExpression;
import oracle.AWXML.HierarchicalParentSourceExpression;
import oracle.AWXML.Hierarchy;
import oracle.AWXML.HierarchyLevelAssociation;
import oracle.AWXML.Level;
import oracle.AWXML.Measure;
import oracle.AWXML.MeasureFolder;
import oracle.AWXML.MeasureSourceExpression;
import oracle.AWXML.PreComputeClause;
import oracle.AWXML.SourceColumn;
import oracle.AWAction.BuildDatabase;
import oracle.AWAction.Interaction;

import oracle.express.spl.SPLExecutor;

import oracle.jdbc.OracleConnection;

import java.io.File;
import java.io.FileWriter;

import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import java.util.Iterator;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * Creates an analytic workspace based on the Global schema and populates
 * the analytic workspace with data from the tables of that relational schema.
 * Before creating the analytic workspace, the program deletes an analytic
 * workspace of the same name if it already exists.
 * The program generates a log file and it creates a file that records the
 * creation of the Analytic Workspace Java API objects as an XML string.
 *
 * This program requires the first version of the Global Schema for Documentation,
 * whichs is available on the Oracle Tecnology Network at
 * http://www.oracle.com/technology/products/bi/olap/olap.html. To download the 
 * sample schema, in the Documentation section see Sample Schemas for Documentation.

 * To run this program, use the command line arguments described in the
 * storeCommandLineProperties method, with global_aw as the -user and
 * -password arguments.
 */
public class BuildAWExample
{
  public static AWConnection awConnection;
  private Interaction curInteraction;
  private Properties props = new Properties();

  // The elements in the _keys Vector are the relational key columns to which
  // a Level is mapped.
  Vector _keys = new Vector(0);
  // The elements in the _cols Vector are the relational columns to which
  // the attributes for a Level are mapped.
  Vector _cols = new Vector(0);
  // The elements in the _levels Vector are the Level objects for specifying
  // the levels of a hierarchy.
  Vector _levels = new Vector(0);
  // The elements in the _levColumns Vector are the relation columns to which
  // the levels of the hierarchy are mapped.
  Vector _levColumns = new Vector(0);
  Vector _dimList = new Vector(0);
  Vector _measures = new Vector(0);
  Vector _measCols = new Vector(0);

  // File name for writing the XML string.
  String xmlStr = "BuildAWExample.xml";

  // File for AW build log.
  String xmlBldStr = "buildAW.log";

  /**
   * Constructor.
   */
  public BuildAWExample()
  {
  }

  /**
   * Creates the top-level objects for the analytic workspace and calls methods
   * that create the lower-level objects.
   * Writes the analytic workspace objects to an XML string.
   * Builds the analytic workspace framework and populates the analytic
   * workspace with data.
   * Connects to the Oracle Database instance, retrieves information
   * about the construction of the analytic workspace, and writes the
   * information to a log file.
   * Finally, closes the connection to the database.
   */
  public void run(String[] args)
  {
    // Store the command line arguments in a Properties object for use in
    // establishing a connection to the Oracle Database instance.
    this.storeCommandLineProperties(args);

    // Get the command line arguments that are stored in the Property object.
    String connection = props.getProperty("conn");
    String user = props.getProperty("user");
    String password = props.getProperty("password");

    // Create the AWConnection.
    try
    {
      System.out.println("Connecting to " + connection + ".");
      awConnection = new AWConnection(user, password, connection);
    }
    catch(Exception e)
    {
      System.out.println("Error connecting to " + connection + ".\n"  + e);
    }

    // Specify a name for the analytic workspace.
    String awName = "GLOBALAW";

    // Delete the analytic workspace if it already exists.
    deleteExistingAW(awConnection, user, awName);

    // Create an AW object and give it a name.
    AW globalAW = new AW();
    globalAW.setName(awName);

    // Create Dimension objects, and pass each Dimension to a method that 
    // creates Attribute, Hierarchy, and Level objects for it.

    // Create the TIME_AW Dimension and set its names.
    Dimension timeDim = globalAW.createDimension();
    timeDim.setName("TIME_AW");
    timeDim.setShortName("TIME_AW");
    timeDim.setLongName("TIME_AW dimension");
    timeDim.setIsTime(true);

    // Create the Attribute, Level, and Hierarchy objects for the Dimension.
    this.createTimeDim(timeDim);

    // Get the detail level to use when creating a Cube.
    Vector _time_levels = timeDim.getLevels();
    Level monthLevel = (Level)_time_levels.elementAt(0);

    // Create the CUSTOMER_AW Dimension and set its names.
    Dimension customerDim = globalAW.createDimension();
    customerDim.setName("CUSTOMER_AW");
    customerDim.setShortName("CUSTOMER_AW");
    customerDim.setLongName("CUSTOMER_AW dimension");

    // Create the Attribute, Level, and Hierarchy objects for the Dimension.
    this.createCustomerDim(customerDim);

    // Get the detail level to use when creating a Cube.
    Vector customerDim_levels = customerDim.getLevels();
    Level shipToLevel = (Level)customerDim_levels.elementAt(0);

    // Create the PRODUCT_AW Dimension and set its names.
    Dimension productDim = globalAW.createDimension();
    productDim.setName("PRODUCT_AW");
    productDim.setShortName("PRODUCT_AW");
    productDim.setLongName("PRODUCT_AW Dimension");

    // Create the Attribute, Level, and Hierarchy objects for the Dimension.
    this.createProductDim(productDim);

    // Get the detail level to use when creating a Cube.
    Vector productDim_levels = productDim.getLevels();
    Level itemLevel = (Level)productDim_levels.elementAt(0);

    // Create the CHANNEL_AW Dimension and set its names.
    Dimension channelDim = globalAW.createDimension();
    channelDim.setName("CHANNEL_AW");
    channelDim.setShortName("CHANNEL_AW");
    channelDim.setLongName("CHANNEL_AW dimension");

    // Create the Attribute, Level, and Hierarchy objects for the Dimension.
    this.createChannelDimension(channelDim);

    // Get the detail level, which is shared by all dimension hierarchies,
    // to use when creating a Cube.
    Vector _channel_levels = channelDim.getLevels();
    Level channelLevel = (Level)_channel_levels.elementAt(0);

    // Create the UNITS_CUBE_AW Cube and its measures.
    this.clearCubeMapVectors();
    Cube unitsCube = globalAW.createCube();
    unitsCube.setName("UNITS_CUBE_AW");

    // Add the dimensions to the _dimList Vector.
    _dimList.add(timeDim);
    _dimList.add(customerDim);
    _dimList.add(productDim);
    _dimList.add(channelDim);

    // Specify the dimensions for the Cube.
    this.associateCubeDimensions(unitsCube, _dimList);

    // Create an aggregation map for the Cube.
    this.precomputeNA(unitsCube, "AGGMAP_UC");

    // Create a Measure for the unit sales quantities.
    Measure units = unitsCube.createMeasure();

    units.setName("UNITS_AW");
    units.setShortName("UNITS_AW");
    units.setLongName("UNITS_AW measure");
    units.setDataType("NUMBER");
    
    // Add the Measure to the _measures Vector.
    _measures.add(units);
    // Add the column of the Global schema relational table to the
    // _measCols Vector.
    _measCols.add("GLOBAL.UNITS_HISTORY_FACT.UNITS");

    // Add the detail levels for the dimensions to the _levels Vector and
    // the relational columns for the levels to the _cols Vector.
    _levels.add(monthLevel);
    _cols.add("GLOBAL.UNITS_HISTORY_FACT.MONTH_ID");
    _levels.add(shipToLevel);
    _cols.add("GLOBAL.UNITS_HISTORY_FACT.SHIP_TO_ID");
    _levels.add(itemLevel);
    _cols.add("GLOBAL.UNITS_HISTORY_FACT.ITEM_ID");
    _levels.add(channelLevel);
    _cols.add("GLOBAL.UNITS_HISTORY_FACT.CHANNEL_ID");

    // Map the columns of the relational tables to the dimensions and measures
    // of the analytic workspace Cube.
    this.createCubeMap(unitsCube, _levels, _cols, _measures, _measCols);

    // Create the PRICE_COST_CUBE_AW Cube and its measures.
    this.clearCubeMapVectors();

    Cube priceCostCube = globalAW.createCube();
    priceCostCube.setName("PRICE_COST_CUBE_AW");

    _dimList.add(productDim);
    _dimList.add(timeDim);

    this.associateCubeDimensions(priceCostCube, _dimList);
    this.precomputeNA(priceCostCube, "AGGMAP_PCC");

    // Create a Measure for unit prices.
    Measure price = priceCostCube.createMeasure();

    price.setName("PRICE_AW");
    price.setShortName("PRICE_AW");
    price.setLongName("PRICE_AW measure");
    price.setDataType("NUMBER");
    _measures.add(price);
    _measCols.add("GLOBAL.PRICE_AND_COST_HISTORY_FACT.UNIT_PRICE");

    // Create a Measure for unit costs.
    Measure cost = priceCostCube.createMeasure();

    cost.setName("COST_AW");
    price.setShortName("COST_AW");
    cost.setLongName("COST_AW measure");
    cost.setDataType("NUMBER");
    _measures.add(cost);
    _measCols.add("GLOBAL.PRICE_AND_COST_HISTORY_FACT.UNIT_COST");

    _levels.add(itemLevel);
    _cols.add("GLOBAL.PRICE_AND_COST_HISTORY_FACT.ITEM_ID");
    _levels.add(monthLevel);
    _cols.add("GLOBAL.PRICE_AND_COST_HISTORY_FACT.MONTH_ID");

    this.createCubeMap(priceCostCube, _levels, _cols, _measures, _measCols);

    // Create a MeasureFolder and add the measures to it.
    // A MeasureFolder appears as an MdmSchema object to an
    // Oracle OLAP Java API application.
    MeasureFolder measureFolder = globalAW.createMeasureFolder();
    measureFolder.setName("GLOBALAW_SCHEMA");
    measureFolder.setShortName("GLOBALAW_SCHEMA");
    measureFolder.setLongName("GLOBALAW_SCHEMA schema");
    measureFolder.addMeasure(units);
    measureFolder.addMeasure(cost);
    measureFolder.addMeasure(price);

    // Write the XML string for the analytic workspace.
    String XML = globalAW.WriteToXML();

    try
    {
      File xmlFile = new File(xmlStr);
      FileWriter xmlout = new FileWriter(xmlFile);
      xmlout.write(XML);
      xmlout.write("\n");
      xmlout.close();
    }
    catch (Exception e)
    {
      System.out.println("Failure to write XML string to file: " +
                          e.toString());
    }

    // Build the analytic workspace framework.
    System.out.println("Building the " + awName + " framework.");
    globalAW.Create(awConnection);
    globalAW.Commit(awConnection);

    System.out.println(awName + " framework built!");

    // Populate the analytic workspace with the metadata objects and 
    // the data from the relational schema.
    System.out.println("Populating " + awName + "." );
    try
    {
      xml_build_global_aw(awName);
    }
    catch(Exception e)
    {
      System.out.println("The following error occurred while building  "
                        + awName + ":\n\t" + e.toString());
    }

    // Connect to the Oracle Database instance and retrieve information
    // about building the analytic workspace.
    try
    {
      // Create a DriverManager.
      DriverManager.registerDriver (new oracle.jdbc.OracleDriver());
      // Create a connection to the Oracle Database instance.
      OracleConnection o_connection =
      (OracleConnection)DriverManager.getConnection("jdbc:oracle:thin:@" +
         connection, user, password);

      // Get information from the olapsys.xml_load_log table.
      String result = null;
      String sqlText = "select distinct xml_loadid " +
                       "from olapsys.xml_load_log " +
                       "order by xml_loadid desc";
      Statement sqlStmt = o_connection.createStatement();
      ResultSet rs = sqlStmt.executeQuery(sqlText);
      if(rs.next())
      {
        // Get the first value.
        int loadid = rs.getInt(1);
        rs.close();

        // Now get the rest of the log without the timestamp for easy diffing.
        sqlText = "select SUBSTR(xml_message, 9) xml_message " +
                  "from olapsys.xml_load_log " +
                  "where xml_loadid = " + loadid + " " +
                  "order by xml_loadid, xml_recordid";
        rs = sqlStmt.executeQuery(sqlText);
        String tempResult;
        while(rs.next())
        {
          tempResult = rs.getString(1);
          if(result == null)
            result = tempResult.trim();
          else
            result = result + "\n" + tempResult.trim();
        }
      }

      // Write the information to a file.
      try
      {
        File xmlBuildLog = new File(xmlBldStr);
        FileWriter xmlBldout = new FileWriter(xmlBuildLog);
        xmlBldout.write(result);
        xmlBldout.write("\n");
        xmlBldout.close();
      }
      catch (Exception e)
      {
        System.out.println("Failure to write XML build log: " + e.toString());
      }
    }
    catch(Exception e)
    {
      System.out.println("Verifier failure with " + e.toString());
    }

    // Close the connection to the database.
    awConnection.close();
  }

  /**
   *  Builds the analytic workspace.
   */
  public void xml_build_global_aw(String awName)
  {
    // Create an Interaction object for building the analytic workspace.
    curInteraction = new Interaction();

    // Set the AWConnection to use when building the analytic workspace.
    curInteraction.setConnection(awConnection);

    // Create an BuildDatabase object and give it a name.
    BuildDatabase myBuild =
                   (BuildDatabase)curInteraction.createAction("BUILDDATABASE");

    // Set the name of the analytic workspace.
    myBuild.setAWName(awName);

    // Specify not running solves when building the database.
    myBuild.setRunSolve(false);

    // Build the analytic workspace.
    myBuild.Execute();
  }

  /**
   * Creates Attribute, Level, and Hierarchy objects for the Time Dimension.
   */
  public void createTimeDim(Dimension timeDim)
  {
    // Indicate that the primary key of the dimension table contains the
    // lowest-level dimension members.
    timeDim.setUseNativeKey(true);

    // Create the Long and Short Description Attribute objects for the Dimension.
    Attribute timeLDAttr = createLongDescAttr(timeDim);
    Attribute timeSDAttr = createShortDescAttr(timeDim);

    // Create the End_Date and Time_Span Attribute objects.
    Attribute timeEndDateAttr = timeDim.createAttribute();
    timeEndDateAttr.setName("End_Date");
    timeEndDateAttr.setClassification("END_DATE");
    Attribute timeTimeSpanAttr = timeDim.createAttribute();
    timeTimeSpanAttr.setName("Time_Span");
    timeTimeSpanAttr.setClassification("TIME_SPAN");

    // Create the Level objects for the Dimension.

    // Create the detail level, MONTH_AW.
    Level monthLevel = timeDim.createLevel();
    monthLevel.setName("MONTH_AW");
    monthLevel.setShortName("MONTH_AW");
    monthLevel.setLongName("MONTH_AW level");

    // Create AttributeProjection objects for the attributes for the
    // MONTH_AW Level.
    this.setupAttributeProjection(monthLevel, "Long Description", timeLDAttr);
    this.setupAttributeProjection(monthLevel, "Short Description", timeSDAttr);
    this.setupAttributeProjection(monthLevel, "End Date", timeEndDateAttr);
    this.setupAttributeProjection(monthLevel, "Time Span", timeTimeSpanAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.TIME_DIM.MONTH_ID");
    _cols.add("GLOBAL.TIME_DIM.MONTH_DSC");  // Long description
    _cols.add("GLOBAL.TIME_DIM.MONTH_DSC");  // Short description
    _cols.add("GLOBAL.TIME_DIM.MONTH_END_DATE");
    _cols.add("GLOBAL.TIME_DIM.MONTH_TIMESPAN");
    this.createLevelMap(monthLevel, _keys,_cols);

    // Create the QUARTER_AW Level.
    Level quarterLevel = timeDim.createLevel();
    quarterLevel.setName("QUARTER_AW");
    quarterLevel.setShortName("QUARTER_AW");
    quarterLevel.setLongName("QUARTER_AW level");

    // Create AttributeProjection objects for the attributes for the
    // QUARTER_AW Level.
    this.setupAttributeProjection(quarterLevel, "Long Description",
                                  timeLDAttr);
    this.setupAttributeProjection(quarterLevel, "Short Description",
                                  timeSDAttr);
    this.setupAttributeProjection(quarterLevel, "End Date", timeEndDateAttr);
    this.setupAttributeProjection(quarterLevel, "Time Span", timeTimeSpanAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.TIME_DIM.QUARTER_ID");
    _cols.add("GLOBAL.TIME_DIM.QUARTER_DSC");  // Long description
    _cols.add("GLOBAL.TIME_DIM.QUARTER_DSC");  // Short description
    _cols.add("GLOBAL.TIME_DIM.QUARTER_END_DATE");
    _cols.add("GLOBAL.TIME_DIM.QUARTER_TIMESPAN");
    this.createLevelMap(quarterLevel, _keys, _cols);

    // Create the top level, YEAR_AW.
    Level yearLevel = timeDim.createLevel();
    yearLevel.setName("YEAR_AW");
    yearLevel.setShortName("YEAR_AW");
    yearLevel.setLongName("YEAR_AW level");

    // Create AttributeProjection objects for the attributes for the
    // YEAR_AW Level.
    this.setupAttributeProjection(yearLevel, "Long Description", timeLDAttr);
    this.setupAttributeProjection(yearLevel, "Short Description", timeSDAttr);
    this.setupAttributeProjection(yearLevel, "End Date", timeEndDateAttr);
    this.setupAttributeProjection(yearLevel, "Time Span", timeTimeSpanAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.TIME_DIM.YEAR_ID");
    _cols.add("GLOBAL.TIME_DIM.YEAR_DSC"); // Long description
    _cols.add("GLOBAL.TIME_DIM.YEAR_DSC"); // Short description
    _cols.add("GLOBAL.TIME_DIM.YEAR_END_DATE");
    _cols.add("GLOBAL.TIME_DIM.YEAR_TIMESPAN");
    this.createLevelMap(yearLevel, _keys, _cols);

    // Create the CALENDAR_AW Hierarchy.
    Hierarchy calendarHier = timeDim.createHierarchy();
    calendarHier.setName("CALENDAR_AW");
    calendarHier.setShortName("CALENDAR_AW");
    calendarHier.setLongName("CALENDAR_AW hierarchy");


    // Add the Level objects for the Hierarchy, and the columns for them,
    // to Vector objects in the hierarcharchical order of the levels,
    // with the top level first.
    this.clearHierarchyMapVectors();
    _levels.add(yearLevel);
    _levColumns.add("GLOBAL.TIME_DIM.YEAR_ID");
    _levels.add(quarterLevel);
    _levColumns.add("GLOBAL.TIME_DIM.QUARTER_ID");
    _levels.add(monthLevel);
    _levColumns.add("GLOBAL.TIME_DIM.MONTH_ID");

    // Specify the levels for the hierarchy and map them to relational columns.
    this.createHierarchyMap(calendarHier, _levels, _levColumns);
  }

  /**
   * Creates Attribute, Hierarchy, and Level objects for the Customer Dimension.
   */
  public void createCustomerDim(Dimension customerDim)
  {
    // Indicate that the primary key of the dimension table contains the
    // lowest-level dimension members.
    customerDim.setUseNativeKey(true);

    // Create the Long and Short Description Attribute objects for the Dimension.
    Attribute customerLDAttr = createLongDescAttr(customerDim);
    Attribute customerSDAttr = createShortDescAttr(customerDim);

    // Create Level objects for the SHIPMENTS_ROLLUP_AW Hierarchy.

    // Create the detail level, SHIP_TO_AW.
    // This level is in both of the hierarchies of the Customer Dimension.
    Level customerShip_ToLevel = customerDim.createLevel();
    customerShip_ToLevel.setName("SHIP_TO_AW");
    customerShip_ToLevel.setShortName("SHIP_TO_AW");
    customerShip_ToLevel.setLongName("SHIP_TO_AW level");


    // Create the Long and Short Description AttributeProjection objects for
    // SHIP_TO_AW Level.
    this.setupAttributeProjection(customerShip_ToLevel, "Long Description",
                                  customerLDAttr);
    this.setupAttributeProjection(customerShip_ToLevel, "Short Description",
                                  customerSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.CUSTOMER_DIM.SHIP_TO_ID");
    _cols.add("GLOBAL.CUSTOMER_DIM.SHIP_TO_DSC"); // Long description
    _cols.add("GLOBAL.CUSTOMER_DIM.SHIP_TO_DSC"); // Short description
    this.createLevelMap(customerShip_ToLevel, _keys, _cols);

    // Create the WAREHOUSE_AW level.
    Level customerWarehouseLevel = customerDim.createLevel();
    customerWarehouseLevel.setName("WAREHOUSE_AW");
    customerWarehouseLevel.setShortName("WAREHOUSE_AW");
    customerWarehouseLevel.setLongName("WAREHOUSE_AW level");

    // Create the Long and Short Description AttributeProjection objects for
    // WAREHOUSE_AW Level.
    this.setupAttributeProjection(customerWarehouseLevel, "Long Description",
                                  customerLDAttr);
    this.setupAttributeProjection(customerWarehouseLevel, "Short Description",
                                  customerSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.CUSTOMER_DIM.WAREHOUSE_ID");
    _cols.add("GLOBAL.CUSTOMER_DIM.WAREHOUSE_DSC"); // Long description
    _cols.add("GLOBAL.CUSTOMER_DIM.WAREHOUSE_DSC"); // Short description
    this.createLevelMap(customerWarehouseLevel, _keys, _cols);

    // Create the REGION_AW Level.
    Level custRegionLevel = customerDim.createLevel();
    custRegionLevel.setName("REGION_AW");
    custRegionLevel.setShortName("REGION_AW");
    custRegionLevel.setLongName("REGION_AW level");

    // Create the Long and Short Description AttributeProjection objects for
    // the REGION_AW Level.
    this.setupAttributeProjection(custRegionLevel, "Long Description",
                                  customerLDAttr);
    this.setupAttributeProjection(custRegionLevel, "Short Description",
                                  customerSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.CUSTOMER_DIM.REGION_ID");
    _cols.add("GLOBAL.CUSTOMER_DIM.REGION_DSC"); // Long description
    _cols.add("GLOBAL.CUSTOMER_DIM.REGION_DSC"); // Short description
    this.createLevelMap(custRegionLevel, _keys, _cols);

    // Create the top level, ALL_CUSTOMERS_AW.
    Level allCustomersLevel = customerDim.createLevel();
    allCustomersLevel.setName("ALL_CUSTOMERS_AW");
    allCustomersLevel.setShortName("ALL_CUSTOMERS_AW");
    allCustomersLevel.setLongName("ALL_CUSTOMERS_AW level");

    // Create the Long and Short Description AttributeProjection objects for
    // the ALL_CUSTOMERS_AW Level.
    this.setupAttributeProjection(allCustomersLevel, "Long Description",
                                  customerLDAttr);
    this.setupAttributeProjection(allCustomersLevel, "Short Description",
                                  customerSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.CUSTOMER_DIM.ALL_CUSTOMERS_ID");
    _cols.add("GLOBAL.CUSTOMER_DIM.ALL_CUSTOMERS_DSC"); // Long description
    _cols.add("GLOBAL.CUSTOMER_DIM.ALL_CUSTOMERS_DSC"); // Short description
    this.createLevelMap(allCustomersLevel, _keys, _cols);

    // Create the SHIPMENTS_ROLLUP_AW Hierarchy.
    Hierarchy shipmentsRollupHier = customerDim.createHierarchy();
    shipmentsRollupHier.setName("SHIPMENTS_ROLLUP_AW");
    shipmentsRollupHier.setShortName("SHIPMENTS_ROLLUP_AW");
    shipmentsRollupHier.setLongName("SHIPMENTS_ROLLUP_AW level");

    // Add the Level objects for the Hierarchy, and the columns for them,
    // to Vector objects in the hierarcharchical order of the levels,
    // with the top level first.
    this.clearHierarchyMapVectors();
    _levels.add(allCustomersLevel);
    _levColumns.add("GLOBAL.CUSTOMER_DIM.ALL_CUSTOMERS_ID");
    _levels.add(custRegionLevel);
    _levColumns.add("GLOBAL.CUSTOMER_DIM.REGION_ID");
    _levels.add(customerWarehouseLevel);
    _levColumns.add("GLOBAL.CUSTOMER_DIM.WAREHOUSE_ID");
    _levels.add(customerShip_ToLevel);
    _levColumns.add("GLOBAL.CUSTOMER_DIM.SHIP_TO_ID");

    // Specify the levels for the hierarchy and map them to relational columns.
    this.createHierarchyMap(shipmentsRollupHier, _levels, _levColumns);

    // Create Level objects for the MARKET_ROLLUP_AW Hierarchy.

    // Create the ACCOUNT_AW Level.
    Level customerAccountLevel = customerDim.createLevel();
    customerAccountLevel.setName("ACCOUNT_AW");
    customerAccountLevel.setShortName("ACCOUNT_AW");
    customerAccountLevel.setLongName("ACCOUNT_AW level");

    // Create the Long and Short Description AttributeProjection objects for
    // the ACCOUNT_AW Level.
    this.setupAttributeProjection(customerAccountLevel, "Long Description",
                                  customerLDAttr);
    this.setupAttributeProjection(customerAccountLevel, "Short Description",
                                  customerSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.CUSTOMER_DIM.ACCOUNT_ID");
    _cols.add("GLOBAL.CUSTOMER_DIM.ACCOUNT_DSC"); // Long description
    _cols.add("GLOBAL.CUSTOMER_DIM.ACCOUNT_DSC"); // Short description
    this.createLevelMap(customerAccountLevel, _keys, _cols);

    // Create the MARKET_SEGMENT_AW Level.
    Level customerMktSegmentLevel = customerDim.createLevel();
    customerMktSegmentLevel.setName("MARKET_SEGMENT_AW");
    customerMktSegmentLevel.setShortName("MARKET_SEGMENT_AW");
    customerMktSegmentLevel.setLongName("MARKET_SEGMENT_AW level");

    // Create the Long and Short Description AttributeProjection objects for
    // the MARKET_SEGMENT_AW Level.
    this.setupAttributeProjection(customerMktSegmentLevel, "Long Description",
                                  customerLDAttr);
    this.setupAttributeProjection(customerMktSegmentLevel, "Short Description",
                                  customerSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.CUSTOMER_DIM.MARKET_SEGMENT_ID");
    _cols.add("GLOBAL.CUSTOMER_DIM.MARKET_SEGMENT_DSC"); // Long description
    _cols.add("GLOBAL.CUSTOMER_DIM.MARKET_SEGMENT_DSC"); // Short description
    this.createLevelMap(customerMktSegmentLevel, _keys, _cols);

    // Create the top level, TOTAL_MARKET_AW.
    Level customerTotalMktLevel = customerDim.createLevel();
    customerTotalMktLevel.setName("TOTAL_MARKET_AW");
    customerTotalMktLevel.setShortName("TOTAL_MARKET_AW");
    customerTotalMktLevel.setLongName("TOTAL_MARKET_AW level");

    // Create AttributeProjection objects for the long and short description
    // attributes for the TOTAL_MARKET_AW Level.
    this.setupAttributeProjection(customerTotalMktLevel, "Long Description",
                                  customerLDAttr);
    this.setupAttributeProjection(customerTotalMktLevel, "Short Description",
                                  customerSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.CUSTOMER_DIM.TOTAL_MARKET_ID");
    _cols.add("GLOBAL.CUSTOMER_DIM.TOTAL_MARKET_DSC"); // Long description
    _cols.add("GLOBAL.CUSTOMER_DIM.TOTAL_MARKET_DSC"); // Short description
    this.createLevelMap(customerTotalMktLevel, _keys, _cols);

    // Create the MARKET_ROLLUP_AW Hierarchy.
    Hierarchy marketRollupHier = customerDim.createHierarchy();
    marketRollupHier.setName("MARKET_ROLLUP_AW");
    marketRollupHier.setShortName("MARKET_ROLLUP_AW");
    marketRollupHier.setLongName("MARKET_ROLLUP_AW hierarchy");

    // Add the Level objects for the Hierarchy, and the columns for them,
    // to Vector objects in the hierarcharchical order of the levels,
    // with the top level first.
    this.clearHierarchyMapVectors();
    _levels.add(customerTotalMktLevel);
    _levColumns.add("GLOBAL.CUSTOMER_DIM.TOTAL_MARKET_ID");
    _levels.add(customerMktSegmentLevel);
    _levColumns.add("GLOBAL.CUSTOMER_DIM.MARKET_SEGMENT_ID");
    _levels.add(customerAccountLevel);
    _levColumns.add("GLOBAL.CUSTOMER_DIM.ACCOUNT_ID");
    _levels.add(customerShip_ToLevel);
    _levColumns.add("GLOBAL.CUSTOMER_DIM.SHIP_TO_ID");

    // Specify the levels for the hierarchy and map them to relational columns.
    this.createHierarchyMap(marketRollupHier, _levels, _levColumns);
  }

  /**
   * Creates Attribute, Level, and Hierarchy objects for the Product Dimension.
   */
  public void createProductDim(Dimension productDim)
  {
    // Indicate that the primary key of the dimension table contains the
    // lowest-level dimension members.
    productDim.setUseNativeKey(true);

    // Create the Attribute objects for the Dimension.
    Attribute packageAttr = productDim.createAttribute();
    packageAttr.setName("PACKAGE_AW");
    packageAttr.setClassification("USER");

    Attribute productLDAttr = createLongDescAttr(productDim);
    Attribute productSDAttr = createShortDescAttr(productDim);
    
    // Create the detail level, ITEM_AW.
    Level itemLevel = productDim.createLevel();
    itemLevel.setName("ITEM_AW");
    itemLevel.setShortName("ITEM_AW");
    itemLevel.setLongName("ITEM_AW level");

    // Create AttributeProjection objects for the attributes for the
    // ITEM_AW Level.
    this.setupAttributeProjection(itemLevel, "PACKAGE_AW", packageAttr);
    this.setupAttributeProjection(itemLevel, "Long Description", productLDAttr);
    this.setupAttributeProjection(itemLevel, "Short Description", productSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.PRODUCT_DIM.ITEM_ID");
    _cols.add("GLOBAL.PRODUCT_DIM.ITEM_PACKAGE_ID");
    _cols.add("GLOBAL.PRODUCT_DIM.ITEM_DSC"); // Long description
    _cols.add("GLOBAL.PRODUCT_DIM.ITEM_DSC"); // Short description
    this.createLevelMap(itemLevel, _keys,_cols);

    // Create the FAMILY_AW level.
    Level familyLevel = productDim.createLevel();
    familyLevel.setName("FAMILY_AW");
    familyLevel.setShortName("FAMILY_AW");
    familyLevel.setLongName("FAMILY_AW level");

    // Create AttributeProjection objects for the attributes for the
    // FAMILY_AW Level.
    this.setupAttributeProjection(familyLevel, "Long Description",
                                  productLDAttr);
    this.setupAttributeProjection(familyLevel, "Short Description",
                                  productSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.PRODUCT_DIM.FAMILY_ID");
    _cols.add("GLOBAL.PRODUCT_DIM.FAMILY_DSC"); // Long description
    _cols.add("GLOBAL.PRODUCT_DIM.FAMILY_DSC"); // Short description
    this.createLevelMap(familyLevel, _keys,_cols);

    // Create the CLASS_AW level.
    Level classLevel = productDim.createLevel();
    classLevel.setName("CLASS_AW");
    classLevel.setShortName("CLASS_AW");
    classLevel.setLongName("CLASS_AW level");

    // Create AttributeProjection objects for the attributes for the
    // CLASS_AW Level.
    this.setupAttributeProjection(classLevel, "Long Description",
                                  productLDAttr);
    this.setupAttributeProjection(classLevel, "Short Description",
                                  productSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.PRODUCT_DIM.CLASS_ID");
    _cols.add("GLOBAL.PRODUCT_DIM.CLASS_DSC"); // Long description
    _cols.add("GLOBAL.PRODUCT_DIM.CLASS_DSC"); // Short description
    this.createLevelMap(classLevel, _keys,_cols);

    // Create the top level, TOTAL_PRODUCT_AW.
    Level totalProductLevel = productDim.createLevel();
    totalProductLevel.setName("TOTAL_PRODUCT_AW");
    totalProductLevel.setShortName("TOTAL_PRODUCT_AW");
    totalProductLevel.setLongName("TOTAL_PRODUCT_AW level");

    // Create AttributeProjection objects for the attributes for the
    // TOTAL_PRODUCT_AW Level.
    this.setupAttributeProjection(totalProductLevel, "Long Description",
                                  productLDAttr);
    this.setupAttributeProjection(totalProductLevel, "Short Description",
                                  productSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.PRODUCT_DIM.TOTAL_PRODUCT_ID");
    _cols.add("GLOBAL.PRODUCT_DIM.TOTAL_PRODUCT_DSC"); // Long description
    _cols.add("GLOBAL.PRODUCT_DIM.TOTAL_PRODUCT_DSC"); // Short description
    this.createLevelMap(totalProductLevel, _keys,_cols);
    
    // Create the PRODUCT_ROLLUP_AW Hierarchy.
    Hierarchy productRollupHier = productDim.createHierarchy();
    productRollupHier.setName("PRODUCT_ROLLUP_AW");
    productRollupHier.setShortName("PRODUCT_ROLLUP_AW");
    productRollupHier.setLongName("PRODUCT_ROLLUP_AW level");


    // Add the Level objects for the Hierarchy, and the columns for them,
    // to Vector objects in the hierarcharchical order of the levels,
    // with the top level first.
    this.clearHierarchyMapVectors();
    _levels.add(totalProductLevel);
    _levColumns.add("GLOBAL.PRODUCT_DIM.TOTAL_PRODUCT_ID");
    _levels.add(classLevel);
    _levColumns.add("GLOBAL.PRODUCT_DIM.CLASS_ID");
    _levels.add(familyLevel);
    _levColumns.add("GLOBAL.PRODUCT_DIM.FAMILY_ID");
    _levels.add(itemLevel);
    _levColumns.add("GLOBAL.PRODUCT_DIM.ITEM_ID");

    // Specify the levels for the hierarchy and map them to relational columns.
    this.createHierarchyMap(productRollupHier, _levels, _levColumns);
  }

  /**
   * Creates Attribute, Level, and Hierarchy objects for the Channel Dimension.
   */
  public void createChannelDimension(Dimension channelDim)
  {
    // Indicate that the primary key of the dimension table contains the
    // lowest-level dimension members.
    channelDim.setUseNativeKey(true);

    // Create the Long and Short Description Attribute objects for the Dimension.
    Attribute channelLDAttr = createLongDescAttr(channelDim);
    Attribute channelSDAttr = createShortDescAttr(channelDim);

    // Create the Level objects for the Dimension.

    // Create the detail level, CHANNEL_AW.
    Level channelLevel = channelDim.createLevel();
    channelLevel.setName("CHANNEL_AW");
    channelLevel.setShortName("CHANNEL_AW");
    channelLevel.setLongName("CHANNEL_AW level");

    // Create AttributeProjection objects for the long and short description
    // attributes for the CHANNEL_AW Level.
    this.setupAttributeProjection(channelLevel, "Long Description",
                                  channelLDAttr);
    this.setupAttributeProjection(channelLevel, "Short Description",
                                  channelSDAttr);

    this.clearLevelMapVectors();
    _keys.add("GLOBAL.CHANNEL_DIM.CHANNEL_ID");
    _cols.add("GLOBAL.CHANNEL_DIM.CHANNEL_DSC"); // Long description
    _cols.add("GLOBAL.CHANNEL_DIM.CHANNEL_DSC"); // Short description
    this.createLevelMap(channelLevel, _keys, _cols);

    // Create the top level, ALL_CHANNELS_AW.
    Level allChannelsLevel = channelDim.createLevel();
    allChannelsLevel.setName("ALL_CHANNELS_AW");
    allChannelsLevel.setShortName("ALL_CHANNELS_AW");
    allChannelsLevel.setLongName("ALL_CHANNELS_AW level");

    // Create the Long and Short Description AttributeProjection objects for
    // the ALL_CHANNELS_AW Level.
    this.setupAttributeProjection(allChannelsLevel, "Long Description",
                                  channelLDAttr);
    this.setupAttributeProjection(allChannelsLevel, "Short Description",
                                  channelSDAttr);

    this.clearLevelMapVectors();

    // The order of the elements in the _cols Vector depends on the order
    // in which the attributes are created.
    // Columns in the relational tables are specified in the form
    // user.tablename.column
    _keys.add("GLOBAL.CHANNEL_DIM.ALL_CHANNELS_ID");
    _cols.add("GLOBAL.CHANNEL_DIM.ALL_CHANNELS_DSC"); // Long description
    _cols.add("GLOBAL.CHANNEL_DIM.ALL_CHANNELS_DSC"); // Short description.
    this.createLevelMap(allChannelsLevel, _keys, _cols);

    // Create the CHANNEL_ROLLUP_AW Hierarchy.
    Hierarchy chanRollupHier = channelDim.createHierarchy();
    chanRollupHier.setName("CHANNEL_ROLLUP_AW");
    chanRollupHier.setShortName("CHANNEL_ROLLUP_AW");
    chanRollupHier.setLongName("CHANNEL_ROLLUP_AW hierarchy");


  // Add the Level objects for the Hierarchy, and the columns for them,
  // to Vector objects in the hierarcharchical order of the levels,
  // with the top level first.
    this.clearHierarchyMapVectors();
    _levels.add(allChannelsLevel);
    _levColumns.add("GLOBAL.CHANNEL_DIM.ALL_CHANNELS_ID");
    _levels.add(channelLevel);
    _levColumns.add("GLOBAL.CHANNEL_DIM.CHANNEL_ID");

    // Specify the levels for the hierarchy and map them to relational columns.
    this.createHierarchyMap(chanRollupHier, _levels, _levColumns);
  }

  /**
   * Create the Long_Description Attribute for a Dimension.
   */
  public Attribute createLongDescAttr(Dimension dim)
  {
    Attribute ldAttr = dim.createAttribute();
    ldAttr.setName("Long_Description");
    ldAttr.setClassification("MEMBER_LONG_DESCRIPTION");

    return ldAttr;
  }

  /**
   * Create the Short_Description Attribute for a Dimension.
   */
  public Attribute createShortDescAttr(Dimension dim)
  {
    Attribute sdAttr = dim.createAttribute();
    sdAttr.setName("Short_Description");
    sdAttr.setClassification("MEMBER_SHORT_DESCRIPTION");

    return sdAttr;
  }

  /**
   * Creates an AttributeProjection for a Level and sets its name and the
   * Attribute to project onto the Level.
   */
  public void setupAttributeProjection(Level _lev, String _name, Attribute _attr)
  {
     AttributeProjection _ap = _lev.createAttributeProjection();
    _ap.setName(_name);
    _ap.setAttribute(_attr);
  }

  /**
   * Clears the Vector objects that are used for mapping a Level object to the
   * columns of a relational table.
   */
  public void clearLevelMapVectors()
  {
    _keys.removeAllElements();
    _cols.removeAllElements();
  }

  /**
   * Creates a DimensionMapGroup that maps a Level to the columns of
   * a relational table.
   */
  public void createLevelMap(Level _lev, Vector _keys, Vector _columns)
  {
    // Map the keys of the relational table to the level.
    DimensionMapGroup _dmg = _lev.createSourceDimensionMapGroup();
    DimensionKeySourceExpression _key = _dmg.CreateKeyMap();

    for (int y=0; y < _keys.size(); y++)
    {
      SourceColumn _keycol = _key.CreateSourceColumn();
      _keycol.setColumn((String)_keys.elementAt(y));
    }

    // Map the columns of the relational tables or views to the
    // AttributeProjection objects.
    int i = 0;
    for(Iterator attrs = _lev.getAttributes().iterator(); attrs.hasNext();)
    {
      AttributeProjection _curAttr = (AttributeProjection)attrs.next();
      AttributeSourceExpression _attrMap = _dmg.CreateAttributeMap();
      _attrMap.setTargetObject(_curAttr);
      SourceColumn _attrCol = _attrMap.CreateSourceColumn();
      _attrCol.setColumn((String)_columns.elementAt(i));
      i++;
    }
  }

  /**
   * Creates the associations between Level objects and a Hierarchy.
   */
  public void createHierarchyMap(Hierarchy _hier, Vector _levels, Vector columns)
    {
      int i = 0;
      for(Iterator _lev = _levels.iterator(); _lev.hasNext();)
      {
        Level _curLev = (Level)_lev.next();
        HierarchyLevelAssociation _hla = _hier.createHierarchyLevelAssociation();
        _hla.setLevel(_curLev);

        DimensionMapGroup _dmg = _hla.createSourceDimensionMapGroup();
        DimensionKeySourceExpression _dkse = _dmg.CreateKeyMap();
        SourceColumn _levColumn = _dkse.CreateSourceColumn();
        _levColumn.setColumn((String)columns.elementAt(i));

        // The first level has no parent.
        // Each subsequent level does have a parent.
        // The first element in the Vector must be the top-most level.
        if (i > 0)
        {
          HierarchicalParentSourceExpression _hpse = _dmg.CreateParentMap();
          SourceColumn _hpsesc = _hpse.CreateSourceColumn();
          _hpsesc.setColumn((String)columns.elementAt(i-1));
        }
        i++;
      }
    }

  /**
   * Clears the Vector objects that are used for specifying the Level objects
   * in a Hierarchy and for mapping the levels to the columns of a relational
   * table.
   */
  public void clearHierarchyMapVectors()
  {
    _levels.removeAllElements();
    _levColumns.removeAllElements();
  }

  /**
   * Clears the Vector objects that are used for specifying the Level objects
   * in a Hierarchy and for mapping the levels to the columns of a relational
   * table.
   */
  public void clearCubeMapVectors()
  {
    _measures.removeAllElements();
    _measCols.removeAllElements();
    _levels.removeAllElements();
    _cols.removeAllElements();
    _dimList.removeAllElements();
  }

  /**
   * Specifies the dimensions for a Cube.
   */
  public void associateCubeDimensions(Cube _cube, Vector _dimList)
  {
    for (int i=0; i < _dimList.size(); i++)
    {
      CubeDimRef _cdr = _cube.createCubeDimRef();
      Dimension _dim = (Dimension)_dimList.elementAt(i);
      _cdr.setDimension(_dim);
    }
  }

  /**
   * Maps the columns of the relational tables to the dimensions and measures
   * of the analytic workspace Cube.
   */
  public void createCubeMap(Cube _cube, Vector _levels, Vector _columns,
                            Vector _measures, Vector _measCols)
  {
    CubeMapGroup _cmg = _cube.createSourceCubeMapGroup();

    // Create the primary key in the fact table.
    for (int y=0; y < _levels.size(); y++)
    {
      CubeDimensionSourceExpression _cdse = _cmg.CreateKeyMap();
      _cdse.addLevel((Level)_levels.elementAt(y));

      // Loop through the associated comma delimited
      // FKey columns in the fact table.
      StringTokenizer _st =
                     new StringTokenizer((String)_columns.elementAt(y), "," );
      while (_st.hasMoreTokens())
      {
        SourceColumn _keycol = _cdse.CreateSourceColumn();
        _keycol.setColumn(_st.nextToken());
      }
    }

    for(int x=0; x < _measures.size(); x++)
    {
      Measure curMeasure = (Measure)_measures.elementAt(x);
      curMeasure.setAutoSolve("NO_AUTO_SOLVE");
      String  curMeascol = (String)_measCols.elementAt(x);
      MeasureSourceExpression _measMap = _cmg.CreateAttributeMap();
      _measMap.setTargetObject(curMeasure);
      SourceColumn _meascol = _measMap.CreateSourceColumn();
      _meascol.setColumn(curMeascol);
    }
  }

  /**
   * Specify excluding NA values when computing values while building the analytic workspace.
   */
  public void precomputeNA(Cube _cube, String _aggdef_name)
  {
    // Rename the default aggregation definition that was implicitly created.
    AggregationDefinition _aggdef =_cube.getDefaultAggregationDefinition();
    _aggdef.setName(_aggdef_name);

    // Remove the default calculation specification.
    _aggdef.getCalculationSpecification().clear();

    Vector _dimrefs = _cube.getCubeDimRefs();

    for (Iterator _d = _dimrefs.iterator(); _d.hasNext();)
    {
      AggregationHierarchySpecification _aggHierSpec =
                             _aggdef.createAggregationHierarchySpecification();
      CubeDimRef _dimref = (CubeDimRef)_d.next();
      Dimension _dim = _dimref.getDimension();
      _aggHierSpec.setDimension(_dim);
      PreComputeClause _precompute_clause =
                                         _aggHierSpec.createPreComputeClause();
      DimensionMemberExpression _dimMemExp =
                          _precompute_clause.createDimensionMemberExpression();
      _dimMemExp.setExpression("ALL");
      _dimMemExp.setExpressionType("EXCLUDE");
    }
  }

  /**
   * Stores in a Properties object the command-line arguments used to
   * establish a connection to an instance of Oracle Database.
   *
   * The command-line arguments specify the server, port, and system
   * identifier, and the username and password to use in creating the
   * connection to the Oracle instance.
   *
   * The arguments have the following form:
   *      -propertyName propertyValue
   *
   * The property names are -conn, -user, and -password.
   *
   * The value for -conn is the name of the server machine running the Oracle
   * Database instance, the number of the port on which Oracle OLAP is
   * listening, and system identifier (SID) of the Oracle instance.
   *
   * Specify the connection, a username, and a password as command-line
   * arguments in the following format:
   *
   *   -conn serverName:portNumber:sid -user username -password password
   *
   * In the -conn specification, replace  "serverName" with the hostname of
   * the server on which the Oracle Database instance is running.
   * Replace "portNumber" with the number of the TCP/IP listener port for
   * the database (which is 1521 by default).
   * Replace "sid" with the SID of the Oracle instance to which
   * you want to connect.
   *
   * An example -conn value is "myOracleServer:1521:orcl".
   *
   * An example of the command-line arguments is the following:
   *  -conn myOracleServer:1521:orcl -user global_aw -password global_aw
   *
   * @param args A String array that contains the command-line arguments
   *             to use for connecting to the Oracle Database instance.
   */
   private void storeCommandLineProperties(String[] args)
   {
     for (int i = 0; i < args.length; i += 2)
     {
         if (i + 1 == args.length)
         {
         throw new IllegalArgumentException("Specify command-line arguments "
                        + "in the form -<PropertyName> <PropertyValue>, " +
                       "with the name and the value separated by whitespace.");
       }
       props.put(args[i].substring(1), args[i + 1]);
     }
   }

  /**
   * Deletes the analytic workspace if it already exists.
   */
  private void deleteExistingAW(AWConnection awConnection, String user,
                                String awName)
  {
    try
    {
      OracleConnection conn = (OracleConnection)
                               awConnection.getSQLConnection();
      SPLExecutor splExec = new SPLExecutor(conn);
      splExec.initialize();

      try
      {
        System.out.println("Deleting " + user + "." + awName + " if it exists.");
        splExec.executeCommand("aw delete " + user + "." + awName);
        System.out.println("Deleted " + user + "." + awName + ".");
      }
      catch(Exception e)
      {
        System.out.println("The " +  user + "." + awName + " analytic workspace "
                           + "either did not exist or it could not be deleted.");
      }

    }
    catch(SQLException e)
    {
      System.out.println("Could not initialize the SPL executor. " + e);
    }
  }

  public static void main(String[] args)
  {
    new BuildAWExample().run(args);
  }
}

